(function(log, utils) {
    'use strict';

    // Constants
    var ID_PREFIX = 'ymail_ctr_id_',
        CONTENTEDITABLE_ID_PREFIX = 'ymail_contenteditable_id_';

    /**
     * Base View Controller prototype
     *
     * Objects created to control specific DOM elements and user interaction
     * between those elements should inherit from this prototype.
     */
    function ViewController() {
        this.el = null;
        this.id = utils.uniqueId(ID_PREFIX);
    }

    /**
     * Create the view's element.
     *
     * This method is not responsible for adding the Node to the document,
     * but instead delegates that responsibility to the caller.
     *
     * @param {HtmlElement} [parentEl] element to append the ViewController's element
     * @return {Node}
     *
     * @abstract
     */
    ViewController.prototype.render = function() {
        throw new Error('Not Implemented by subclass');
    };

    ViewController.prototype.attach = function(el) {
        var existingId;

        this.el = el;

        existingId = this.el.getAttribute('id');

        if (existingId && existingId.length > 0) {
            this.id = this.el.getAttribute('id');
        }

        this.el.setAttribute('id', this.id);
    };

    /**
     * Inserts the controller's element into the document
     * at the current selection.
     *
     * Selection range must be set before calling.  If selection
     * range is not set (rangeCount === 0), then this function will
     * not insert anything into the DOM.
     *
     * If insertElement is set, insert the given element insert object el. It is
     * used when a view object needs a wrapper(Like inline image).
     */
    ViewController.prototype.renderIntoSelection = function(isCardElement, insertElement) {
        var contentEditableIds,
            sel,
            range,
            rangeCount;

        // If el is null or undefined
        if (!this.el) {
            return;
        }

        sel = window.getSelection();
        rangeCount = sel.rangeCount;
        range = rangeCount > 0 && sel.getRangeAt && sel.getRangeAt(0);

        // Old Webkit based WebView's have selection issues
        // unfortunately this means we must fall back to using execCmd hackery
        if (!range) {
            log.d('[ViewController]: getRangeAt is not supported.  Falling back to execCommand');

            contentEditableIds = new Map();

            // Query for existing HtmlElements with contenteditable attribute
            utils.forEach(this.el.querySelectorAll('[contenteditable]'), function(el) {
                var id = utils.uniqueId(CONTENTEDITABLE_ID_PREFIX);

                // set an id to allow lookup after insertion
                el.setAttribute('id', id);

                // cache the attribute value.
                // we can then later reset this value to make sure it persists after insertion.
                contentEditableIds.set(id, el.getAttribute('contenteditable'));
            });

            /*
             * Insert
             *
             * WARNING: execCommand may remove attributes,
             * one of which is the contenteditable=false
             */
            if (!document.execCommand('insertHTML', false, insertElement ? insertElement.outerHTML : this.el.outerHTML)) {
                log.w('[ViewController]: Fail to insert, execCommand insertHtml failed.  Most likely caused by no current selection range.');
                return;
            }

            this.el = document.getElementById(this.id);

            // Set the contenteditable attribute to make sure execCommand did not remove it
            contentEditableIds.forEach(function(value, id) {
                this.el.getElementById(id).setAttribute('contenteditable', value);
            });
        } else {
            if (rangeCount === 0) {
                // No current selection
                log.w('[ViewController]: Fail to insert, no current selection range');
                return;
            }
            log.d('[ViewController]: Inserting Node using Range API');

            if (range.startContainer.nodeName === "BR" ) {
                var newRange = document.createRange();
                newRange.setStartBefore(range.startContainer);
                newRange.insertNode(insertElement ? insertElement : this.el);
            } else if (range.startContainer && range.startContainer.className && range.startContainer.className.indexOf(CARD_EL_CLASS_NAME) >= 0) {
                // If current range is inside a card element, set the range after the card
                range.setStartAfter(range.startContainer);
                range.insertNode(insertElement ? insertElement : this.el);
            } else {
                // When we insert multiple card elements, we don't want to clear the previous card in the range which is the only element
                if (rangeCount > 1) {
                    range.deleteContents();
                }
                range.insertNode(insertElement ? insertElement : this.el);
            }

            if (isCardElement) {
                var newRange = document.createRange();
                newRange.setStartAfter(insertElement ? insertElement : this.el);
                newRange.insertNode(document.createElement("br"));

                // Clear previous selection and set newRange as selection
                sel.removeAllRanges();
                sel.addRange(newRange);
            }
        }
    };

    window.ViewController = ViewController;
})(window.log, window.utils);
